#include "./playbackrate.hpp"

#include "debug/log.hpp"

#include <boost/range/numeric.hpp>

namespace Vape {

PlaybackRateMonitor::PlaybackRateMonitor()
    : started(false)
    , multiplier(1.0f)
    , clamp({-0.1f, 0.1f})
    , aggressiveClamp({-0.1f, 0.1f})
    , minOffset(0.001f)
    , useAggressive(false)
{
}

PlaybackRateMonitor::~PlaybackRateMonitor()
{
}

float
PlaybackRateMonitor::CalculatePlaybackRateFromOffset(float offset)
{
    float absOffset = std::abs(offset);
    bool isNeg      = offset < 0;

    if (absOffset < 0.2f) {
        this->useAggressive = false;
    }

    if (absOffset < this->minOffset) {
        offset = 0;
    }

    offset = std::pow(absOffset, 2);
    offset *= this->multiplier;

    if (isNeg) {
        offset *= -1.f;
    }

    auto c = this->clamp;
    if (this->useAggressive) {
        c = this->aggressiveClamp;
    }

    offset = std::clamp(offset, c.lo, c.hi);

    return 1.0f - offset;
}

void
PlaybackRateMonitor::SetToStarted()
{
    this->started = true;
}

void
PlaybackRateMonitor::SetOffsetMultiplier(float mul)
{
    this->multiplier = mul;
}

void
PlaybackRateMonitor::SetOffsetClamp(float lo, float hi)
{
    this->clamp = {lo, hi};
}

void
PlaybackRateMonitor::SetAggressiveOffsetClamp(float lo, float hi)
{
    this->aggressiveClamp = {lo, hi};
}

void
PlaybackRateMonitor::SetMinOffset(float min)
{
    this->minOffset = min;
}

void
PlaybackRateMonitor::UseAggressive()
{
    this->useAggressive = true;
}

PTSRewriter::PTSRewriter(flicks _frameDuration)
    : frameDuration(_frameDuration)
{
}

PTSRewriter::~PTSRewriter()
{
}

int
PTSRewriter::ScheduleFrame(float speed, flicks duration, int maxCount,
                           std::function<void(flicks)> cb)
{
    int count = 0;

    auto fDuration = toFloat(duration);
    fDuration /= speed;
    duration = toFlicks(fDuration);

    this->lastOutputPTS += duration;

    while (count < maxCount && this->lastOutputPTS > this->lastAlignedOutputPTS) {
        cb(this->lastAlignedOutputPTS);
        this->lastAlignedOutputPTS += this->frameDuration;
        count++;
    }

    return count;
}

void
OffsetTracker::SetDesiredDelay(flicks delay)
{
    std::unique_lock lock(this->totalDelayMutex);

    this->desiredDelay = delay;
}

void
OffsetTracker::SetAdditionalDelay(flicks delay)
{
    std::unique_lock lock(this->totalDelayMutex);

    this->additionalDelay = delay;
}

void
OffsetTracker::SetMinTotalDelay(flicks newMinTotalDelay)
{
    std::unique_lock lock(this->totalDelayMutex);

    this->minTotalDelay = newMinTotalDelay;
}

flicks
OffsetTracker::GetTotalDelay() const
{
    std::unique_lock lock(this->totalDelayMutex);

    Log::Trace("Min total delay: {}s", toFloat(this->minTotalDelay));
    return std::clamp(this->desiredDelay + this->additionalDelay, this->minTotalDelay,
                      this->maxTotalDelay);
}

flicks
OffsetTracker::GetIdealPlayheadTimestamp() const
{
    flicks now        = Now();
    flicks totalDelay = this->GetTotalDelay();

    flicks playhead = now - totalDelay;

    return playhead;
}

}  // namespace Vape
