#include "./soundtouchwrapper.hpp"

#include "debug/log.hpp"

extern "C" {
#include <libavutil/samplefmt.h>
#include <libavutil/timestamp.h>
}

#include <cassert>
#include <vector>

using namespace soundtouch;

namespace Vape {

SoundTouchWrapper::SoundTouchWrapper(int _sampleRate, int _noChannels, int _channelLayout)
    : sampleRate(_sampleRate)
    , noChannels(_noChannels)
    , channelLayout(_channelLayout)
{
    this->st.setSampleRate(sampleRate);
    this->st.setChannels(noChannels);

    this->st.setSetting(SETTING_USE_QUICKSEEK, true);
    this->st.setSetting(SETTING_USE_AA_FILTER, true);
}

SoundTouchWrapper::~SoundTouchWrapper()
{
}

void
SoundTouchWrapper::SetTempo(float tempo)
{
    this->st.setTempo(tempo);
}

bool
SoundTouchWrapper::SendAVFrame(AVFrame *in)
{
    if (in->format != AV_SAMPLE_FMT_FLTP) {
        Log::Error("SoundTouchWrapper::SendAVFrame called with wrong sample format");
        av_frame_unref(in);
        return false;
    }

    if (in->sample_rate != this->sampleRate) {
        Log::Error("SoundTouchWrapper::SendAVFrame called with wrong sample rate");
        av_frame_unref(in);
        return false;
    }

    std::vector<float> interleavedData(in->channels * in->nb_samples);
    auto iPtr = interleavedData.data();

    // Convert planar data to interleaved
    for (int i = 0; i < in->nb_samples; ++i) {
        for (int c = 0; c < in->channels; ++c) {
            float *data = reinterpret_cast<float *>(in->data[c]);
            *(iPtr++)   = data[i];
        }
    }

    this->st.putSamples(interleavedData.data(), in->nb_samples);

    this->inputPTS = in->pts;

    av_frame_unref(in);
    return true;
}

bool
SoundTouchWrapper::ReceiveAVFrame(AVFrame *out)
{
    if (out == nullptr) {
        return false;
    }

    const int maxSamples = 2048;

    std::vector<float> data(maxSamples * this->noChannels);

    int samples = this->st.receiveSamples(data.data(), maxSamples);

    if (samples == 0) {
        return false;
    }

    // Set frame metadata
    out->nb_samples     = samples;
    out->channel_layout = this->channelLayout;
    out->channels       = this->noChannels;
    out->format         = AV_SAMPLE_FMT_FLT;
    out->sample_rate    = this->sampleRate;

    // Allocate buffers in frame
    if (av_frame_get_buffer(out, 0) != 0) {
        Log::Error("Could not allocate AVFrame buffer in SoundTouchWrapper");
        return false;
    }

    float *d = reinterpret_cast<float *>(out->data[0]);
    std::copy(data.begin(), data.begin() + (samples * this->noChannels), d);

    out->pts = this->outputPTS;
    this->outputPTS += samples;

    return true;
}

void
SoundTouchWrapper::GetFrames(const AVRational &tb, std::function<bool(AVFrame *, flicks)> cb)
{
    AVFrame *frame = av_frame_alloc();

    while (this->ReceiveAVFrame(frame)) {
        // Calculate how much we deviate from the target offset
        auto tc  = frame->pts;
        auto pts = FlicksFromAVTime(tc, tb);

        bool res = cb(frame, pts);

        av_frame_unref(frame);

        if (!res) {
            break;
        }
    }

    av_frame_free(&frame);
}

flicks
SoundTouchWrapper::GetRewriteOffset() const
{
    // Calculate the accumulated offset between the input and output time code over time, after tempo changes has been applied
    int64_t ptsOffset = this->inputPTS - this->outputPTS - this->st.numUnprocessedSamples();
    return FlicksFromAVTime(ptsOffset, AVRational{1, this->sampleRate});
}

}  // namespace Vape
