#include "manager.hpp"

#include "../app.hpp"
#include "../constants.hpp"
#include "pubsub/settings-parameters.hpp"
#include "vapour-player-core/time.hpp"

#include <boost/algorithm/clamp.hpp>

namespace Vape {

SettingsManager::SettingsManager(PubSub::Client &client)
    // Player settings
    : enabled(client, PubSub::PATH_PREFIX_TOKEN + "/output-0/enabled")
    , streamName(client, PubSub::PATH_PREFIX_TOKEN + "/output-0/streamName")
    , desiredDelay(client, PubSub::PATH_PREFIX_TOKEN + "/output-0/desiredDelay")
    , maxTranscode(client, PubSub::PATH_PREFIX_TOKEN + "/output-0/maxTranscode")
    , outputDeviceName(client, PubSub::PATH_PREFIX_TOKEN + "/output-0/deckLink/outputDeviceName")
    , displayMode(client, PubSub::PATH_PREFIX_TOKEN + "/output-0/displayMode")
    , fallbackFrame(client, PubSub::PATH_PREFIX_TOKEN + "/output-0/fallbackFrame")
    , resetTrigger(client, PubSub::PATH_PREFIX_TOKEN + "/output-0/reset")
    // Decoder settings
    , channelOffset(client, PubSub::PATH_PREFIX_TOKEN + "/output-0/channelOffset")
    , triggerAggressiveRecovery(client,
                                PubSub::PATH_PREFIX_TOKEN + "/output-0/triggerAggressiveRecovery")
    // DeckLink settings
    , genlockOffset(client, PubSub::PATH_PREFIX_TOKEN + "/output-0/deckLink/genlockOffset")
    , SDI3GMode(client, PubSub::PATH_PREFIX_TOKEN + "/output-0/deckLink/SDI3GMode")
{
    // Helper lambda to subscribe to setting and store the listener
    auto subscribe = [this](auto &setting, auto fn) {
        this->listeners.push_back(setting.signal.AddDirectListener(fn));
    };

    /**
     * Player setting: Enabled
     */
    subscribe(this->enabled, [this](auto &, auto &v) {
        this->state.player.enabled  = v;
        this->playerSettingsChanged = true;
    });

    /**
     * Player setting: Stream Name
     */
    subscribe(this->streamName, [this](auto &, auto &v) {
        this->state.player.streamName = v;
        this->playerSettingsChanged   = true;
    });

    /**
     * Player setting: Desired delay
     */
    subscribe(this->desiredDelay, [this](auto &, auto &v) {
        this->state.player.desiredDelay = toFlicks(v);
        this->playerSettingsChanged     = true;
    });

    this->desiredDelay.SetParameters({
        MIN_DESIRED_DELAY,
        MAX_DESIRED_DELAY,
        0.25f,
        std::nullopt,
    });

    /**
     * Player setting: Max transcode
     */
    subscribe(this->maxTranscode, [this](auto &, auto &v) {
        PlayerCore::TranscodeLimits limits;

        if (v == "720p") {
            limits.width  = 1280;
            limits.height = 720;
        } else if (v == "900p") {
            limits.width  = 1600;
            limits.height = 900;
        } else if (v == "1080p") {
            limits.width  = 1920;
            limits.height = 1080;
        }

        this->state.player.maxTranscode = limits;
        this->playerSettingsChanged     = true;
    });

    this->maxTranscode.SetOptions({
        {"Disabled", ""},
        {"1080p", "1080p"},
        {"900p", "900p"},
        {"720p", "720p"},
    });

    /**
     * Player setting: Restart playback
     */
    subscribe(this->resetTrigger,
              [this](auto &, auto &, auto) { this->playerSettingsChanged = true; });

    /**
     * Consumer setting: Device name
     */
    subscribe(this->outputDeviceName, [this](auto &, auto &v) {
        this->state.output.outputDeviceString = v;
        this->consumerSettingsChanged         = true;
    });

    this->outputDeviceName.SetOptions({});

    /**
     * Consumer setting: Display mode
     */
    subscribe(this->displayMode, [this](auto &, auto &v) {
        this->state.output.displayModeString = v;
        this->consumerSettingsChanged        = true;
    });

    this->displayMode.SetOptions({});

    /**
     * Consumer setting: Fallback frame
     */
    subscribe(this->fallbackFrame, [this](auto &, auto &v) {
        this->state.output.placeholderTypeString = v;
        this->consumerSettingsChanged            = true;
    });

    this->fallbackFrame.SetOptions({
        {"Glitch", "glitch"},
        {"100% bars", "bars100"},
        {"75% bars", "bars75"},
        {"Black", "black"},
    });

    /**
     * Decoder setting: Channel offset
     */
    subscribe(this->channelOffset, [this](auto &, auto &v) {
        this->state.decoder.channelOffset = v;
        this->decoderSettingsChanged      = true;
    });

    /**
     * Decoder setting: Trigger aggressive recovery
     */
    subscribe(this->triggerAggressiveRecovery,
              [this](auto &, auto &, auto) { this->aggressiveRecoverySignal.Emit(); });

    /**
     * DeckLink setting: Genlock offset
     */
    subscribe(this->genlockOffset, [this](auto &, auto &v) {
        this->state.deckLink.genlockOffset = v;
        this->deckLinkSettingsChanged      = true;
    });

    /**
     * DeckLink setting: 3G-SDI mode
     */
    subscribe(this->SDI3GMode, [this](auto &, auto &v) {
        this->state.deckLink.sdi3GMode = v;
        this->deckLinkSettingsChanged  = true;
    });
}

bool
SettingsManager::Poll()
{
    bool needsReset = false;

    if (this->playerSettingsChanged) {
        this->playerSettingsSignal.Emit(this->state.player);
        this->playerSettingsChanged = false;
        needsReset                  = true;
    }

    if (this->consumerSettingsChanged) {
        this->consumerSettingsSignal.Emit(this->state.output);
        this->consumerSettingsChanged = false;
        needsReset                    = true;
    }

    if (this->decoderSettingsChanged) {
        this->decoderSettingsSignal.Emit(this->state.decoder);
        this->decoderSettingsChanged = false;
    }

    if (this->deckLinkSettingsChanged) {
        this->deckLinkSettingsSignal.Emit(this->state.deckLink);
        this->deckLinkSettingsChanged = false;
    }

    return needsReset;
}

void
SettingsManager::SetDeviceNames(const std::vector<std::string> &names)
{
    std::vector<std::pair<std::string, std::string>> options;
    for (auto &name : names) {
        options.emplace_back(name, name);
    }
    this->outputDeviceName.SetOptions(options);
}

void
SettingsManager::SetDisplayModes(const DisplayModeNames &modeNames)
{
    this->displayMode.SetOptions(modeNames);
}

bool
SettingsManager::IsEnabled() const
{
    return this->state.player.enabled && !this->state.player.streamName.empty();
}

}  // namespace Vape
