#pragma once
#include "../consumer/ivideooutputconsumer.hpp"
#include "./consumer.hpp"
#include "./decklink.hpp"
#include "./decoder.hpp"
#include "./null-value.hpp"
#include "./player.hpp"
#include "fundamentals/signal-listener.hpp"
#include "fundamentals/signal-provider.hpp"
#include "pubsub/client.hpp"
#include "pubsub/settings.hpp"

#include <atomic>

namespace Vape {
class Application;

/**
 * Manages all settings
 * Stores the most current values in its corresponding struct
 * Provides a Poll function called from the main thread to fetch updates
 */
class SettingsManager
{
    Signal::ListenerPtrList listeners;

    // Player settings
    PubSub::Settings::Setting<bool> enabled;
    PubSub::Settings::Setting<std::string> streamName;
    PubSub::Settings::Setting<float> desiredDelay;
    PubSub::Settings::Setting<std::string> maxTranscode;
    PubSub::Settings::Setting<std::string> outputDeviceName;
    PubSub::Settings::Setting<std::string> displayMode;
    PubSub::Settings::Setting<std::string> fallbackFrame;
    PubSub::Settings::SignalValueSubscriber<NullValue> resetTrigger;

    // Decoder settings
    PubSub::Settings::Setting<float> channelOffset;
    PubSub::Settings::SignalValueSubscriber<NullValue> triggerAggressiveRecovery;

    // DeckLink settings
    PubSub::Settings::Setting<int> genlockOffset;
    PubSub::Settings::Setting<bool> SDI3GMode;

    // Bools to track settings changes since last poll
    bool playerSettingsChanged   = false;
    bool consumerSettingsChanged = false;
    bool decoderSettingsChanged  = false;
    bool deckLinkSettingsChanged = false;

public:
    // Stores the most recent parameters
    struct State {
        PlayerSettings player;
        ConsumerSettings output;
        DecoderSettings decoder;
        DeckLinkSettings deckLink;
    } state;

    SettingsManager(PubSub::Client &);

    /**
     * Poll if there has been any changes since last poll
     * Changes are fired as siognals
     * @return True if playback needs to be restarted as a consequence of a changed setting
     */
    bool Poll();

    void SetDeviceNames(const std::vector<std::string> &names);
    void SetDisplayModes(const DisplayModeNames &modeNames);

    bool IsEnabled() const;

    Signal::Provider<PlayerSettings> playerSettingsSignal;
    Signal::Provider<ConsumerSettings> consumerSettingsSignal;
    Signal::Provider<DecoderSettings> decoderSettingsSignal;
    Signal::Provider<DeckLinkSettings> deckLinkSettingsSignal;
    Signal::Provider<> aggressiveRecoverySignal;
};
}  // namespace Vape
