#include "./manager.hpp"

#include "../consumer/decklink-output.hpp"
#include "../decoder/libav-decoder.hpp"
#include "../player/player.hpp"

using namespace std::chrono_literals;

namespace Vape {
using namespace Topic;
using namespace PubSub;

template <typename T>
inline TopicGroup<T>::TopicGroup(PubSub::Client &client_, std::string topic_,
                                 std::optional<std::string> path_, system_clock::duration tick_)
    : provider(client_, topic_, path_)
    , tick(tick_)
    , lastUpdate(system_clock::duration{0})
    , changed(false)
{
}

template <typename T>
inline void
TopicGroup<T>::Poll(system_clock::time_point tp)
{
    if ((this->lastUpdate + this->tick) > tp) {
        return;
    }

    this->changed |= this->value.Update(tp);

    if (this->changed) {
        this->provider.SetValue(this->value);
        this->changed    = false;
        this->lastUpdate = tp;
    }
}

template <typename T>
template <typename... Params>
inline void
TopicGroup<T>::Reset(const Params &... params)
{
    this->changed |= this->value.Reset(params...);
}

template <typename T>
template <typename Member, typename ValueType>
inline void
TopicGroup<T>::UpdateMember(Member T::*c, const ValueType &v)
{
    auto &member = this->value.*c;
    if (member == v) {
        return;
    }

    member        = v;
    this->changed = true;
}

template <typename T>
template <typename L>
inline void
TopicGroup<T>::Update(L l)
{
    this->changed |= l(this->value);
}

inline TopicGroup<Topic::MultistreamPlaybackStatus> &
StatusManager::GetStreamStatus(StreamIndex index)
{
    return (index == StreamIndex::Video) ? videoStatus : audioStatus;
}

StatusManager::StatusManager(PubSub::Client &pubsubClient)
    : deckLink(pubsubClient, MultistreamDeckLink::topicString, PATH_PREFIX_TOKEN, 1s)
    , audioLevels(pubsubClient, MultistreamAudioLevels::topicString, PATH_PREFIX_TOKEN, 300ms)
    , videoStatus(pubsubClient, MultistreamPlaybackStatus::videoTopicString, PATH_PREFIX_TOKEN,
                  500ms)
    , audioStatus(pubsubClient, MultistreamPlaybackStatus::audioTopicString, PATH_PREFIX_TOKEN,
                  500ms)
    , systemStatus(pubsubClient, MultistreamSystemStatus::topicString, PATH_PREFIX_TOKEN, 1s)
{
}

template <typename L, typename... S>
inline void
StatusManager::Subscribe(Signal::ListenerPtrList &list, Signal::Provider<S...> &s, L l)
{
    list.push_back(s.AddIndirectListener(this->signalGroup, l));
}

void
StatusManager::SubscribeToTwitchStreamFetcher(Signal::ListenerPtrList &list,
                                              PlayerCore::Listener &s)
{
    Subscribe(list, s.activeTranscodeSignal, [this](const auto &name, const auto isSource) {
        this->systemStatus.UpdateMember(&MultistreamSystemStatus::activeTranscode, name);
        this->systemStatus.UpdateMember(&MultistreamSystemStatus::activeTranscodeIsSource,
                                        isSource);
    });

    Subscribe(list, s.minTotalDelaySignal, [this](const auto minTotalDelay) {
        float fDelay = toFloat(minTotalDelay);
        this->systemStatus.UpdateMember(&MultistreamSystemStatus::minTotalDelay, fDelay);
    });

    Subscribe(list, s.statusSignal, [this](const auto status, const auto type) {
        this->systemStatus.UpdateMember(&MultistreamSystemStatus::streamStatus, status);
        this->systemStatus.UpdateMember(&MultistreamSystemStatus::streamType, type);
    });
}

void
StatusManager::SubscribeToDecoder(Signal::ListenerPtrList &list, LibAVDecoder &d)
{
    // Subscribe to audio levels
    Subscribe(list, d.audioPeakSignal(), bind_this(this, &StatusManager::HandleAudioLevels));

    // Handle buffer length signal from decoder, containg both the new value, and the target value
    Subscribe(list, d.decodedBufferLengthSignal, [this](auto index, auto val, auto target) {
        auto &status = this->GetStreamStatus(index);
        status.Update([val, target](auto &playbackStatus) {
            auto &frameBuffer = playbackStatus.frameBuffer;

            float fTarget = toFloat(target);
            float fMin    = 0;
            float fMax    = fTarget * 2.f;
            float fVal    = toFloat(val);
            if (frameBuffer.target != fTarget || frameBuffer.min != fMin ||
                frameBuffer.max != fMax || frameBuffer.value != fVal) {
                frameBuffer.min    = fMin;
                frameBuffer.target = fTarget;
                frameBuffer.max    = fMax;
                frameBuffer.value  = fVal;
                return true;
            }

            return false;
        });
    });

    // Subscribe to buffer length signal
    Subscribe(list, d.sampleBufferLengthSignal, [this](auto index, auto val) {
        this->GetStreamStatus(index).UpdateMember(&MultistreamPlaybackStatus::sampleBuffer,
                                                  toFloat(val));
    });

    // Subscribe to active total delay signal
    Subscribe(list, d.totalDelaySignal, bind_this(this, &StatusManager::HandleTotalDelay));

    // Subscribe to PTS search signal
    Subscribe(list, d.ptsSearchDoneSignal, [this]() {
        this->systemStatus.UpdateMember(&MultistreamSystemStatus::ptsSearchDone, true);
    });

    Subscribe(list, d.timeBaseCorrectedSignal(), [this](bool val) {
        this->systemStatus.UpdateMember(&MultistreamSystemStatus::timeBaseCorrected, val);
    });

    Subscribe(list, d.startedSignal, [this]() {
        this->systemStatus.UpdateMember(&MultistreamSystemStatus::playbackStarted, true);
    });

    Subscribe(list, d.speedSignal, [this](auto index, auto speed) {
        this->GetStreamStatus(index).UpdateMember(&MultistreamPlaybackStatus::speed, speed);
    });

    Subscribe(list, d.offsetSignal, [this](auto index, auto offset) {
        this->GetStreamStatus(index).UpdateMember(&MultistreamPlaybackStatus::offset, offset);
    });

    Subscribe(list, d.decodeProcessingTimeSignal, [this](auto index, const auto &record) {
        auto &perfMon = this->perfMons[static_cast<int>(index)];
        perfMon.Push(record);
        float saturation = perfMon.GetCPUSaturation();
        this->GetStreamStatus(index).UpdateMember(&MultistreamPlaybackStatus::cpuSaturation,
                                                  saturation);
    });
}

void
StatusManager::SubscribeToDeckLink(Signal::ListenerPtrList &list, DeckLinkOutput &d)
{
    this->Subscribe(list, d.genlockStatusSignal, [this](auto status) {
        this->deckLink.UpdateMember(&MultistreamDeckLink::genlockStatus, status);
    });
}

void
StatusManager::Poll()
{
    auto now = system_clock::now();

    this->signalGroup.Poll();

    this->deckLink.Poll(now);
    this->audioLevels.Poll(now);
    this->videoStatus.Poll(now);
    this->audioStatus.Poll(now);

    auto readyForLive =
        this->videoStatus.Value().IsReadyForLive() && this->audioStatus.Value().IsReadyForLive();
    this->systemStatus.UpdateMember(&MultistreamSystemStatus::readyForLive, readyForLive);

    this->systemStatus.Poll(now);
}

void
StatusManager::Reset(bool enabled)
{
    this->audioAvg.clear();
    this->audioLevels.Reset();
    this->systemStatus.Reset(enabled);
    this->audioStatus.Reset();
    this->videoStatus.Reset();
    this->perfMons[0].Reset();
    this->perfMons[1].Reset();
}

void
StatusManager::HandleAudioLevels(const std::vector<float> &channels)
{
    auto LinearTodB = [](float x) {
        if (x == 0) {
            return -96.33f;
        }

        return std::log10(x) * 20.f;
    };

    // If audio channel count changes, resize vector of QueueAverage
    this->audioAvg.resize(channels.size());

    std::vector<Topic::MultistreamAudioLevels::Level> levels;

    int i = 0;
    for (auto &ch : channels) {
        // Get the name of the channel
        const char *name = i == 0 ? "L" : "R";

        auto &avgChannel = this->audioAvg[i++];

        // Push the new value on the channel's queue and calculate new peak and average
        avgChannel.Push(ch);
        auto avg  = avgChannel.Average();
        auto peak = avgChannel.Max();

        // Push channel to levels vector
        levels.push_back({LinearTodB(avg), LinearTodB(peak), std::move(name), 0});
    }

    // Update topic with the new value
    this->audioLevels.UpdateMember(&Topic::MultistreamAudioLevels::levels, levels);
}

void
StatusManager::HandleTotalDelay(const flicks &delay)
{
    float fDelay = toFloat(delay);
    float fMax   = fDelay * 2.f;

    this->systemStatus.UpdateMember(&MultistreamSystemStatus::activeTotalDelay, fDelay);

    auto updateStream = [fDelay, fMax](auto &streamStatus) {
        float warnMin = fDelay * 0.7f;
        float warnMax = fDelay * 1.5f;
        float errMin  = fDelay * 0.5f;
        float errMax  = fDelay * 2.f;

        streamStatus.sampleBuffer.Init(0, fMax, fDelay);
        streamStatus.sampleBuffer.SetWarningBounds(warnMin, warnMax);
        streamStatus.sampleBuffer.SetErrorBounds(errMin, errMax);
        return true;
    };

    this->videoStatus.Update(updateStream);
    this->audioStatus.Update(updateStream);
}

}  // namespace Vape
