#pragma once

#include "./statusindicator.hpp"

namespace Vape {

template <typename T>
struct MultistreamPlaybackStatusGroup {
    bool
    operator==(const T o) const
    {
        return this->value == o;
    }

    void
    operator=(const T v)
    {
        this->value = v;
    }

    void
    Init(const T _min, const T _max, const T _target)
    {
        this->min    = _min;
        this->max    = _max;
        this->target = _target;
    }

    void
    SetWarningBounds(T min, T max)
    {
        this->warningBounds.Set(min, max);
    }

    void
    SetErrorBounds(T min, T max)
    {
        this->errorBounds.Set(min, max);
    }

    bool
    Update(system_clock::time_point now)
    {
        if (!this->errorBounds.CheckBounds(this->value)) {
            this->indicator.Set(StatusIndicator::Level::Error, now);
        } else if (!this->warningBounds.CheckBounds(this->value)) {
            this->indicator.Set(StatusIndicator::Level::Warning, now);
        } else {
            this->indicator.Set(StatusIndicator::Level::OK, now);
        }

        return this->indicator.Update(now);
    }

    bool
    Reset()
    {
        bool changed = this->value != T();
        changed |= this->indicator.Reset();
        this->value = T();
        return changed;
    }

    T value  = T();
    T min    = -1;
    T max    = 1;
    T target = 0;
    StatusIndicator indicator{std::chrono::seconds{1}};

private:
    struct Bounds {
        T min, max;
        bool set = false;

        void
        Set(const T _min, const T _max)
        {
            this->min = _min;
            this->max = _max;
            this->set = true;
        }

        bool
        CheckBounds(const T v) const
        {
            bool res = (!this->set || (v > this->min && v < this->max));
            return res;
        }
    };

    Bounds warningBounds;
    Bounds errorBounds;
};
}  // namespace Vape

namespace pajlada {
template <typename T, typename RJValue>
struct Serialize<Vape::MultistreamPlaybackStatusGroup<T>, RJValue> {
    static RJValue
    get(const Vape::MultistreamPlaybackStatusGroup<T> &in, typename RJValue::AllocatorType &a)
    {
        RJValue out(rapidjson::kObjectType);

        Vape::rj::Set(out, "Value", in.value, a);
        Vape::rj::Set(out, "Min", in.min, a);
        Vape::rj::Set(out, "Max", in.max, a);
        Vape::rj::Set(out, "Target", in.target, a);
        Vape::rj::Set(out, "Status", in.indicator, a);

        return out;
    }
};
}  // namespace pajlada
