#pragma once
#include "./multistream-playbackstatus-group.hpp"
#include "fundamentals/helpers.hpp"
#include "fundamentals/rapidjson-helpers.hpp"
#include "vapour-player-core/time.hpp"

#include <pajlada/serialize/serialize.hpp>

#include <tuple>

namespace Vape::Topic {
/**
 * The topic for DeckLink status reporting
 */
struct MultistreamPlaybackStatus {
    static constexpr const char *videoTopicString{"MultistreamVideoPlaybackStatus"};
    static constexpr const char *audioTopicString{"MultistreamAudioPlaybackStatus"};

    MultistreamPlaybackStatusGroup<float> cpuSaturation;
    MultistreamPlaybackStatusGroup<float> sampleBuffer;
    MultistreamPlaybackStatusGroup<float> frameBuffer;
    MultistreamPlaybackStatusGroup<float> offset;
    MultistreamPlaybackStatusGroup<float> speed;

    MultistreamPlaybackStatus()
    {
        this->cpuSaturation.Init(0, 1.0f, 0.6f);
        this->cpuSaturation.SetWarningBounds(0, 0.8f);
        this->cpuSaturation.SetErrorBounds(0, 0.9f);

        this->offset.Init(-5.f, 5.0f, 0);
        this->offset.SetWarningBounds(-0.25f, 0.25f);
        this->offset.SetErrorBounds(-0.5f, 0.5f);

        this->speed.Init(0.7f, 1.3f, 1.f);
        this->speed.SetWarningBounds(0.95f, 1.05f);
        this->speed.SetErrorBounds(0.9f, 1.1f);
    }

    bool
    Reset()
    {
        bool changed = this->cpuSaturation.Reset();
        changed |= this->sampleBuffer.Reset();
        changed |= this->frameBuffer.Reset();
        changed |= this->offset.Reset();
        changed |= this->speed.Reset();
        return changed;
    }

    bool
    Update(system_clock::time_point now)
    {
        bool updated = false;
        updated |= this->cpuSaturation.Update(now);
        updated |= this->sampleBuffer.Update(now);
        updated |= this->frameBuffer.Update(now);
        updated |= this->offset.Update(now);
        updated |= this->speed.Update(now);
        return updated;
    }

    bool
    IsReadyForLive() const
    {
        if (this->offset.indicator.IsErrorOrDisabled()) {
            return false;
        } else if (this->speed.indicator.IsErrorOrDisabled()) {
            return false;
        } else if (this->sampleBuffer.indicator.IsErrorOrDisabled()) {
            return false;
        } else if (this->frameBuffer.indicator.IsErrorOrDisabled()) {
            return false;
        } else if (this->cpuSaturation.indicator.IsErrorOrDisabled()) {
            return false;
        }

        return true;
    }
};

}  // namespace Vape::Topic

namespace pajlada {
template <typename RJValue>
struct Serialize<Vape::Topic::MultistreamPlaybackStatus, RJValue> {
    static RJValue
    get(const Vape::Topic::MultistreamPlaybackStatus &in, typename RJValue::AllocatorType &a)
    {
        RJValue out(rapidjson::kObjectType);

        Vape::rj::Set(out, "CPUSaturation", in.cpuSaturation, a);
        Vape::rj::Set(out, "SampleBuffer", in.sampleBuffer, a);
        Vape::rj::Set(out, "FrameBuffer", in.frameBuffer, a);
        Vape::rj::Set(out, "Offset", in.offset, a);
        Vape::rj::Set(out, "Speed", in.speed, a);

        return out;
    }
};
}  // namespace pajlada
