#include "./statusindicator.hpp"

#include "debug/log.hpp"

namespace Vape {
StatusIndicator::StatusIndicator(system_clock::duration _holdDuration)
    : holdDuration(_holdDuration)
    , timePoints({system_clock::time_point{system_clock::duration{0}}})
    , latest(Level::Disabled)
    , activeLevel(Level::Disabled)
{
}

StatusIndicator::Level
StatusIndicator::operator()() const
{
    return this->activeLevel;
}

void
StatusIndicator::Set(Level level, system_clock::time_point now)
{
    this->latest                              = level;
    this->timePoints[static_cast<int>(level)] = now;
    //Log::Debug("StatusIndicator::= {}", level);
}

bool
StatusIndicator::operator==(const Level level) const
{
    return this->latest == level;
}

bool
StatusIndicator::operator!=(StatusIndicator &o)
{
    return this->Tie() == o.Tie();
}

bool
StatusIndicator::Reset()
{
    StatusIndicator test(this->holdDuration);
    bool changed = *this != test;
    if (changed) {
        this->Tie() = test.Tie();
        return true;
    }

    return false;
}

bool
StatusIndicator::Update(system_clock::time_point now)
{
    auto threshold = now - this->holdDuration;

    Level newLevel = Level::Disabled;

    // Test if there is a change to a level that is newer than the threshold
    auto test = [this](auto testLevel, auto threshold) {
        return this->timePoints[static_cast<int>(testLevel)] > threshold;
    };

    // Levels are evaluated in the order of most to least severe
    if (test(Level::Error, threshold)) {
        newLevel = Level::Error;
    } else if (test(Level::Warning, threshold)) {
        newLevel = Level::Warning;
    } else if (test(Level::OK, threshold)) {
        newLevel = Level::OK;
    }

    // Return true if value has changed
    if (this->activeLevel != newLevel) {
        this->activeLevel = newLevel;
        return true;
    }

    return false;
}

bool
StatusIndicator::IsErrorOrDisabled() const
{
    return this->activeLevel == Level::Error || this->activeLevel == Level::Disabled;
}

}  // namespace Vape
