#pragma once

#include "fundamentals/helpers.hpp"
#include "fundamentals/rapidjson-helpers.hpp"
#include "vapour-player-core/time.hpp"

#include <string>
#include <tuple>

namespace Vape {
class StatusIndicator
{
public:
    enum class Level {
        Disabled = 0,
        OK       = 1,
        Warning  = 2,
        Error    = 3,
        Count    = 4,
    };

    StatusIndicator(system_clock::duration _holdDuration);

    void Set(Level level, system_clock::time_point now);
    bool operator==(const Level level) const;
    bool operator!=(StatusIndicator &o);
    Level operator()() const;
    bool Reset();
    bool Update(system_clock::time_point now);
    bool IsErrorOrDisabled() const;

private:
    system_clock::duration holdDuration;
    std::array<system_clock::time_point, static_cast<size_t>(Level::Count)> timePoints;
    Level latest;
    Level activeLevel;

    auto
    Tie()
    {
        return std::tie(holdDuration, timePoints, latest, activeLevel);
    }
};

inline std::ostream &
operator<<(std::ostream &os, StatusIndicator::Level v)
{
    switch (v) {
        case StatusIndicator::Level::Disabled:
            return os << "Disabled";
        case StatusIndicator::Level::OK:
            return os << "OK";
        case StatusIndicator::Level::Warning:
            return os << "Warn";
        case StatusIndicator::Level::Error:
            return os << "Err";
        default:
            return os << "InternalError:" << static_cast<int>(v);
    }
}

}  // namespace Vape

namespace pajlada {
template <typename RJValue>
struct Serialize<Vape::StatusIndicator, RJValue> {
    static RJValue
    get(const Vape::StatusIndicator &in, typename RJValue::AllocatorType &a)
    {
        auto s = fS("{}", in());
        RJValue out(s.c_str(), s.length(), a);

        return out;
    }
};
}  // namespace pajlada
