#!/bin/sh

temp=$(getopt -o : -l docker,release,relwithdebinfo,clang-tidy,clean,env-file:,asan,msan,tsan,usan,manta,separate-symbols -- "$@")

eval set -- "$temp"

set -e

cmake_args=""
manta="0"
separate_symbols="0"
build_type="debug"
build_dir="build"

script_path=$(CDPATH='' cd -- "$(dirname -- "$0")" && pwd)
. "$script_path/config.sh"

while true; do
    case "$1" in
        --release ) build_type="release"; shift ;;
        --relwithdebinfo ) build_type="relwithdebinfo"; shift ;;
        --docker )
            echo " * Building in Docker mode (different output directories)"
            build_dir="$build_dir-docker"
            shift ;;
        --clang-tidy )
            echo " * Enabling clang-tidy"
            cmake_args="$cmake_args -DENABLE_CLANG_TIDY=1"
            shift ;;
        --clean ) clean="1"; shift ;;
        --env-file ) env_file="$2"; shift 2 ;;
        --asan )
            echo " * Enabling Address Sanitizer"
            cmake_args="$cmake_args -DSANITIZE_ADDRESS=1"
            shift ;;
        --msan )
            echo " * Enabling Memory Sanitizer"
            cmake_args="$cmake_args -DSANITIZE_MEMORY=1"
            shift ;;
        --tsan )
            echo " * Enabling Thread Sanitizer"
            cmake_args="$cmake_args -DSANITIZE_THREAD=1"
            shift ;;
        --usan )
            echo " * Enabling Undefined Sanitizer"
            cmake_args="$cmake_args -DSANITIZE_UNDEFINED=1"
            shift ;;
        --manta )
            echo " * Building in Manta mode (different output directories)"
            manta="1"
            shift ;;
        --separate-symbols )
            separate_symbols="1"
            shift ;;
        -- ) shift; break ;;
        * ) break ;;
    esac
done

if [ -n "$env_file" ]; then
    set -a
    . "$env_file"
    set +a
elif [ -f "$HOME/env/build" ]; then
    set -a
    . "$HOME/env/build"
    set +a
fi

if [ -z "$CC" ]; then
    export CC=/usr/bin/clang
fi
if [ -z "$CXX" ]; then
    export CXX=/usr/bin/clang++
fi

build_dir="$build_dir-$build_type"

if [ "$manta" = "1" ]; then
    # Override the build directory to always be the same, regardless of build type or flags.
    # This is fine since we're building inside a docker instance with fresh folders
    build_dir="build-manta"
fi

echo "Build dir: $build_dir"

if [ "$clean" = "1" ]; then
    rm -rf "$build_dir"
fi

if [ ! -d "$build_dir" ]; then
    mkdir "$build_dir"
fi
cd "$build_dir"
cmake_args="$cmake_args -DCMAKE_BUILD_TYPE=$build_type"
cmake $cmake_args ..
make -j

if [ "$separate_symbols" = "1" ]; then
	find \
        "$bin_folder" \
        -maxdepth 1 \
        -not -name "*.debug" \
        -executable \
        -type f \
        -exec sh -c 'file -b "$1" | >/dev/null grep -F "with debug_info"' _ {} \; \
        -printf ' * Separating symbols from %p to %p.debug..\n' \
        -exec objcopy --only-keep-debug '{}' '{}.debug' \; \
        -exec strip --strip-debug --strip-unneeded '{}' \;
fi
