import { createSocket, Socket } from 'dgram';
import { fromBuffer, OSCBaseMessage, OSCBundle, OSCMessage } from 'osc-min';
import { Logger } from 'winston';

export class OSCSocket {
  /**
   * The UDP socket
   */
  private server?: Socket;

  /**
   * Constructor
   * @param port The port to listen to
   * @param host The host to listen to
   * @param logger A Winston logger instance
   */
  public constructor(
    private port: number,
    private host: string,
    private logger: Logger,
    private cb: (msg: OSCMessage) => void
  ) {}

  /**
   * Creates the socket and binds it to an interface
   */
  public async Listen(): Promise<void> {
    // Make sure the previous socket is closed
    await this.Close();

    return new Promise((resolve, reject) => {
      try {
        // Create the socket
        this.server = createSocket('udp4', msg => this.ParseMessage(msg));

        // Bind it to an interface and port
        this.server.bind({ address: this.host, port: this.port }, () => {
          this.logger.info(`OSC socket bound to ${this.host}:${this.port}`);
          resolve();
        });
      } catch (e) {
        reject(e);
      }
    });
  }

  /**
   * Close the socket if it exists
   */
  public async Close(): Promise<void> {
    return new Promise(resolve => {
      if (this.server) {
        this.server.close(resolve);
        this.server = undefined;
      } else {
        resolve();
      }
    });
  }

  /**
   * Parse a message from a buffer and emit and event with the address as event name
   * @param buf The buffer to parse
   */
  private ParseMessage(buf: Buffer) {
    try {
      // Parse the buffer
      const msg = fromBuffer(buf);
      this.HandleBaseMessage(msg);
    } catch (e) {
      this.logger.error('Received a bad OSC message');
    }
  }

  private HandleBaseMessage(msg: OSCBaseMessage) {
    switch (msg.oscType) {
      case 'message': {
        // This is a regular message, send it to the callback
        this.cb(msg as OSCMessage);
        break;
      }
      case 'bundle': {
        // This is a bundle of messages (or potentially a bundle of bundles),
        // repeat process for each element
        const bundle = msg as OSCBundle;
        bundle.elements.forEach(e => this.HandleBaseMessage(e));
        break;
      }
      default:
        this.logger.error(`Unknown OSC message type '${msg.oscType}'`);
    }
  }
}
