import { OSCMessage, OSCStringValue, OSCValue } from 'osc-min';
import { IClientInfo, IClientListTopic } from 'vapour-output-base';
import { VapourServiceBase } from 'vapour-service-base';
import { LoggerFactory } from 'vapour-ts-api-tools';

import { OSCSocket } from './classes/socket';
import { Version } from './version';

const HOST: string = process.env.VAPOUR_OSCSERVICE_HOST || '0.0.0.0';
const PORT: number = process.env.VAPOUR_OSCSERVICE_PORT
  ? parseInt(process.env.VAPOUR_OSCSERVICE_PORT)
  : 3333;

export class OSCService extends VapourServiceBase {
  private oscSocket: OSCSocket;

  public constructor() {
    super('osc-service', Version, {
      logger: LoggerFactory('OSCService', 'cyanBright'),
      rules: [
        // {
        //   Effect: 'Allow',
        //   Path: '/clients/multistream/*/output-0/streamName',
        //   Scope: 'Publish',
        //   Topic: 'SetValue',
        // },
      ],
    });

    // Set up the OSCSocket instance
    this.oscSocket = new OSCSocket(PORT, HOST, this.logger, msg => this.HandleOSCMessage(msg));

    // Subscribe to ClientList
    this.AddListener('ClientList', []);
  }

  protected async StartServiceLogic(): Promise<void> {
    // Start OSC socket
    await this.oscSocket.Listen();
  }

  protected StopServiceLogic(): void {
    // Close the OSC socket
    this.oscSocket.Close();
  }

  /**
   * Handle a new message from the OSCSocket
   * @param msg
   */
  private HandleOSCMessage(msg: OSCMessage): void {
    // Split the address into its components
    const components: string[] = msg.address.split('/');
    if (components.length !== 4) {
      return;
    }

    // Get the components
    const type = components[1];
    const name = components[2];
    const action = components[3];

    // Get the ClientList
    const clientList = this.GetListenerValue<IClientListTopic>('ClientList');
    if (!clientList) {
      return;
    }

    // Find the client in the list
    const client = clientList.Clients.find(info => info.Type === type && info.Name === name);

    if (!client) {
      this.logger.warn(`Unknown client name '${name}' in the OSC address`);
      return;
    }

    switch (type) {
      case 'multistream':
        return this.HandleMultistreamAction(client, action, msg.args);
      default:
        this.logger.error(`Unknown client type '${type}' in OSC address`);
    }
  }

  /**
   * Handle an action with the /multistream type
   * @param client The IClientInfo to perform the action on
   * @param action Name of the action to perform
   * @param args The arguments from the OSC message
   */
  private HandleMultistreamAction(client: IClientInfo, action: string, args: OSCValue[]): void {
    // Select the action
    switch (action) {
      case 'streamName':
        return this.SetStreamName(client, args);
      default: {
        this.logger.error(`Unknown multistream action '${action}' in OSC address`);
      }
    }
  }

  /**
   * Handle a streamName action
   * @param client The IClientInfo to perform the action on
   * @param args The arguments from the OSC message
   */
  private SetStreamName(client: IClientInfo, args: OSCValue[]): void {
    // Check that we get arguments of the correct types
    if (args.length !== 1 || args[0].type !== 'string') {
      this.logger.error(`SetStreamName called with invalid argument types`);
      return;
    }

    // Extract the streamName
    const streamName = (args[0] as OSCStringValue).value;

    // Send the value to Core
    this.vapour.SetValue(`/clients/${client.UUID}/output-0/streamName`, streamName);
    this.logger.info(`Updated streamName for multistream '${client.Name}' to '${streamName}'`);
  }
}
