import 'cross-fetch/polyfill';

import ApolloClient from 'apollo-boost';
import {get} from 'lodash';
import {TypeDesc} from 'vapour-connect/dist';
import {TriggerWS} from 'vapour-connect/dist/type';
import {VapourServiceBase} from 'vapour-service-base';

import {GetActiveEventVapour, ParticipantInput, UpdateEventParticipantsVapour} from './gql-types';
import Version from './version';

const VapourExtensionAuthorization: string|undefined =
    process.env.SECRET_VAPOUR_RIVALSEXTENSION_AUTHORIZATION;
const VapourExtensionChannelID: string|undefined = process.env.VAPOUR_RIVALSEXTENSION_CHANNEL_ID;

const endpoint = 'https://rivals-v6.twitch-extension-backend.com/graphql';

class RivalsExtensionService extends VapourServiceBase {
  triggerUpdatePlayerData: TriggerWS;
  apolloClient = new ApolloClient({
    uri: endpoint,
    headers: {
      authorization: VapourExtensionAuthorization,
    },
  });

  constructor() {
    super('rivals-extension-service', Version);

    this.AddListener('UserChannelData');

    this.vapour.ws.setValuePersistent(
        this.ClientBasePath() + '/channelID', VapourExtensionChannelID);
    this.vapour.ws.setValuePersistent(
        this.ClientBasePath() + '/authorizationIsSet', !!VapourExtensionAuthorization);

    this.triggerUpdatePlayerData = this.typeManager.GetConnectedTrigger(
        TypeDesc.Boolean, this.ClientBasePath() + '/updatePlayerData',
        () => this.UpdatePlayerData().then(console.log).catch(console.error));
  }

  ProduceData(): any {}

  async UpdatePlayerData() {
    const activeEvent = await this.apolloClient.query({
      query: GetActiveEventVapour,
      variables: {
        channelId: VapourExtensionChannelID,
      },
      fetchPolicy: 'no-cache',
    });

    const event = get(activeEvent, 'data.channel.currentEvent');
    const eventId = get(event, 'eventId');
    const participants = get(event, 'participants');

    if (!event || typeof eventId !== 'string' || !Array.isArray(participants)) {
      throw 'GetActiveEvent data malformed';
    }

    const topic = this.GetListenerValue('UserChannelData');

    if (!topic) {
      throw 'UserChannelData malformed';
    }

    const channels = topic.Channels;

    if (!Array.isArray(channels)) {
      throw 'SheetData not an array';
    }

    const participantUpdates: ParticipantInput[] =
        participants.map((participant): ParticipantInput => {
          const participantId = get(participant, 'participantId');

          if (!participantId || typeof participantId !== 'string') {
            throw 'Participant missing participantId';
          }

          const res: ParticipantInput = {participantId, metaVapour: {}};

          const findPlayer = channels.find(row => get(row, 'Channel ID') === participantId);

          if (findPlayer) {
            res.metaVapour.moneyTotal = String(get(findPlayer, 'Player Score', ''));

            const rank = parseInt(get(findPlayer, 'Team Rank'), 10);
            if (isFinite(rank)) {
              res.metaVapour.teamRank = rank;
            }
          }

          return res;
        });

    return this.apolloClient.mutate({
      mutation: UpdateEventParticipantsVapour,
      variables: {
        'channelId': VapourExtensionChannelID,
        'eventId': eventId,
        'participants': participantUpdates,
      },
    });
  }
}

const service = new RivalsExtensionService();
