#include "thread-helpers/thread-helpers.hpp"

#ifdef _WIN32
#include <windows.h>
const DWORD MS_VC_EXCEPTION = 0x406D1388;
#endif

#ifdef __linux__
#include <pthread.h>
#include <sys/prctl.h>
#endif

namespace Vape {

namespace Thread {

int
SetAffinity(std::thread::native_handle_type nativeHandle, int index)
{
#ifdef __linux__
    auto num_cpus = std::thread::hardware_concurrency();

    cpu_set_t cpuset;
    CPU_ZERO(&cpuset);
    CPU_SET(index % num_cpus, &cpuset);

    return pthread_setaffinity_np(nativeHandle, sizeof(cpu_set_t), &cpuset);
#elif _WIN32
    // Thread affinity unimplemented on Windows
    return -1;
#endif
}

int
GetAffinity(std::thread::native_handle_type nativeHandle, cpu_set_t &cpuset)
{
#ifdef __linux__
    CPU_ZERO(&cpuset);
    return pthread_getaffinity_np(nativeHandle, sizeof(cpu_set_t), &cpuset);
#elif _WIN32
    // Thread affinity unimplemented on Windows
    return -1;
#endif
}

}  // namespace Thread

#ifdef _WIN32
#pragma pack(push, 8)
typedef struct tagTHREADNAME_INFO {
    DWORD dwType;      // Must be 0x1000.
    LPCSTR szName;     // Pointer to name (in user addr space).
    DWORD dwThreadID;  // Thread ID (-1=caller thread).
    DWORD dwFlags;     // Reserved for future use, must be zero.
} THREADNAME_INFO;
#pragma pack(pop)

static void
SetThreadName(uint32_t dwThreadID, const char *threadName)
{
    // DWORD dwThreadID = ::GetThreadId( static_cast<HANDLE>( t.native_handle() ) );

    THREADNAME_INFO info;
    info.dwType     = 0x1000;
    info.szName     = threadName;
    info.dwThreadID = dwThreadID;
    info.dwFlags    = 0;

    __try {
        RaiseException(MS_VC_EXCEPTION, 0, sizeof(info) / sizeof(ULONG_PTR), (ULONG_PTR *)&info);
    } __except (EXCEPTION_EXECUTE_HANDLER) {
    }
}
#endif

void
RenameThread(std::string threadName)
{
#ifdef __linux__
    prctl(PR_SET_NAME, threadName.c_str(), 0, 0, 0);
#elif _WIN32
    SetThreadName(GetCurrentThreadId(), threadName.c_str());
#endif
}

}  // namespace Vape
