import FormData from 'form-data';
import nock from 'nock';

import { TicketVendor } from '../throttling/ticket-vendor';
import { timeDifference } from '../time/difference';
import { Validator } from '../validation';
import { APIBase, RequestConfig } from './api-base';

interface IBaseMessage {
  readonly root: true;
}

interface IErrorMessage {
  readonly error: string;
  readonly code: number;
}

interface IApiToken {
  readonly accessToken: string;
}

class ApiConsumer extends APIBase<IBaseMessage, IErrorMessage, IApiToken> {
  protected SetAuthParams(config: RequestConfig, token?: IApiToken): void {
    config.headers = {
      ...(config.headers as Record<string, string>),
      Authorization: `Bearer ${token?.accessToken}`,
    };
  }
}

describe('ApiBase', () => {
  const host = 'https://localhost';
  const path = '/api';
  const requestEndpoint = host + path;

  const schema = {
    definitions: {
      error: {
        properties: {
          code: { type: 'number' },
          error: { type: 'string' },
        },
        required: ['code', 'error'],
        type: 'object',
      },
      root: {
        properties: {
          root: { type: 'boolean' },
        },
        required: ['root'],
        type: 'object',
      },
    },
  };

  describe('construction', () => {
    describe('jsonValidator parameter', () => {
      it('should not construct without a value', () => {
        expect(() => new ApiConsumer(undefined as never)).toThrow(TypeError);
      });

      it('should not construct with an invalid value', () => {
        expect(() => new ApiConsumer('' as never)).toThrow(TypeError);
        expect(() => new ApiConsumer({} as never)).toThrow(TypeError);
        expect(() => new ApiConsumer(true as never)).toThrow(TypeError);
        expect(() => new ApiConsumer(undefined as never)).toThrow(TypeError);
        expect(() => new ApiConsumer(null as never)).toThrow(TypeError);
      });

      it('should construct with a valid value', () => {
        expect(() => new ApiConsumer(new Validator({}))).not.toThrow();
      });
    });

    describe('errorRef parameter', () => {
      let validator!: Validator;
      let throttler!: TicketVendor;

      beforeEach(() => {
        validator = new Validator({});
        throttler = new TicketVendor({ simultaneous: 50, spacing: 0 });
      });

      it('should construct without a value', () => {
        expect(() => new ApiConsumer(validator, throttler, undefined as never)).not.toThrow();
      });

      it('should not construct with an invalid value', () => {
        expect(() => new ApiConsumer(validator, throttler, '' as never)).toThrow(Error);
        expect(() => new ApiConsumer(validator, throttler, {} as never)).toThrow(TypeError);
        expect(() => new ApiConsumer(validator, throttler, true as never)).toThrow(TypeError);
        expect(() => new ApiConsumer(validator, throttler, null as never)).toThrow(TypeError);
      });

      it('should construct with a valid value', () => {
        expect(
          () =>
            new ApiConsumer(validator, new TicketVendor({ simultaneous: 50, spacing: 0 }), 'abc')
        ).not.toThrow();
      });
    });
  });

  describe('methods', () => {
    let instance!: ApiConsumer;
    let validator!: Validator;

    beforeAll(() => {
      validator = new Validator(schema);
    });

    beforeEach(() => (instance = new ApiConsumer(validator, undefined, 'error')));

    describe('#SetAuthParams()', () => {
      it('should get called when making a request', () => {
        const SetAuthParamsSpy = jest.spyOn(instance, 'SetAuthParams' as never);
        const tokenRef = { accessToken: 'baz' } as const;

        instance['TypeCheckedGet']('foo', 'bar', tokenRef);

        expect(SetAuthParamsSpy).toHaveBeenCalled();
        expect(SetAuthParamsSpy.mock.calls[0][1]).toBe(tokenRef);
      });
    });

    //#region GET
    describe('#TypeCheckedGet()', () => {
      const mockRes = { root: true } as IBaseMessage;

      it('should perform a HTTP get request', async () => {
        nock(host).get(path).reply(200);

        const response = await instance['TypeCheckedGet'](requestEndpoint, 'root', {
          accessToken: '',
        });

        expect(response.code).toBe(200);
      });

      it('should return the mocked response', async () => {
        nock(host).get(path).reply(200, mockRes);

        const response = await instance['TypeCheckedGet'](requestEndpoint, 'root', {
          accessToken: '',
        });

        expect(response.data?.root).toBe(true);
        expect(response.code).toBe(200);
        expect(response.error).toBeUndefined();
      });

      it('should validate the error against the schema', async () => {
        nock(host)
          .get(path)
          .reply(500, { code: 1, error: 'test' } as IErrorMessage);

        const response = await instance['TypeCheckedGet'](requestEndpoint, 'root', {
          accessToken: '',
        });

        expect(response.error).not.toBeUndefined();
      });

      it('should call #SetAuthParams()', () => {
        const SetAuthParamsSpy = jest.spyOn(instance, 'SetAuthParams' as never);
        const tokenRef = { accessToken: 'baz' } as const;
        instance['TypeCheckedGet']('foo', 'bar', tokenRef);
        expect(SetAuthParamsSpy).toHaveBeenCalled();
        expect(SetAuthParamsSpy.mock.calls[0][1]).toBe(tokenRef);
      });
    });
    //#endregion

    //#region FORM POST
    describe('#TypeCheckedPostForm()', () => {
      const mockRes = { root: true } as IBaseMessage;

      it('should perform a HTTP post request', async () => {
        const data = new FormData();
        data.append('mockKey', 'mockValue');

        nock(host)
          .post(path, data.getBuffer().toString('utf8'))
          .matchHeader('content-type', data.getHeaders()['content-type'])
          .reply(200, mockRes);

        const response = await instance['TypeCheckedPostForm'](requestEndpoint, data, 'root', {
          accessToken: '',
        });

        expect(response.code).toBe(200);
      });

      it('should return the mocked response', async () => {
        const data = new FormData();
        data.append('mockKey', 'mockValue');

        nock(host)
          .post(path, data.getBuffer().toString('utf8'))
          .matchHeader('content-type', data.getHeaders()['content-type'])
          .reply(200, mockRes);

        const response = await instance['TypeCheckedPostForm'](requestEndpoint, data, 'root', {
          accessToken: '',
        });

        expect(response.data?.root).toBe(true);
        expect(response.code).toBe(200);
        expect(response.error).toBeUndefined();
      });

      it('should validate the error against the schema', async () => {
        const data = new FormData();
        data.append('mockKey', 'mockValue');

        nock(host)
          .post(path, data.getBuffer().toString('utf8'))
          .matchHeader('content-type', data.getHeaders()['content-type'])
          .reply(500, { code: 1, error: 'test' } as IErrorMessage);

        const response = await instance['TypeCheckedPostForm'](requestEndpoint, data, 'root', {
          accessToken: '',
        });

        expect(response.error).not.toBeUndefined();
      });

      it('should call #SetAuthParams()', () => {
        const data = new FormData();
        const SetAuthParamsSpy = jest.spyOn(instance, 'SetAuthParams' as never);
        const tokenRef = { accessToken: 'baz' } as const;

        instance['TypeCheckedPostForm']('foo', data, 'bar', tokenRef);
        expect(SetAuthParamsSpy).toHaveBeenCalled();
        expect(SetAuthParamsSpy.mock.calls[0][1]).toBe(tokenRef);
      });
    });
    //#endregion POST FORM

    //#region URLENCODED POST
    describe('#TypeCheckedPostFormURLENC()', () => {
      const mockRes = { root: true } as IBaseMessage;

      it('should perform a HTTP post request', async () => {
        const data = { mockKey: 'mockValue' } as const;

        nock(host).post(path, data).reply(200, mockRes);

        const response = await instance['TypeCheckedPostFormUrlEnc'](
          requestEndpoint,
          data,
          'root',
          {
            accessToken: '',
          }
        );

        expect(response.code).toBe(200);
      });

      it('should return the mocked response', async () => {
        const data = { mockKey: 'mockValue' } as const;

        nock(host).post(path, data).reply(200, mockRes);

        const response = await instance['TypeCheckedPostFormUrlEnc'](
          requestEndpoint,
          data,
          'root',
          {
            accessToken: '',
          }
        );

        expect(response.data?.root).toBe(true);
        expect(response.code).toBe(200);
        expect(response.error).toBeUndefined();
      });

      it('should validate the error against the schema', async () => {
        const data = { mockKey: 'mockValue' } as const;

        nock(host)
          .post(path, data)
          .reply(500, { code: 1, error: 'test' } as IErrorMessage);

        const response = await instance['TypeCheckedPostFormUrlEnc'](
          requestEndpoint,
          data,
          'root',
          { accessToken: '' }
        );

        expect(response.error).not.toBeUndefined();
      });

      it('should call #SetAuthParams()', () => {
        const data = { mockKey: 'mockValue' } as const;
        const SetAuthParamsSpy = jest.spyOn(instance, 'SetAuthParams' as never);
        const tokenRef = { accessToken: 'baz' } as const;

        instance['TypeCheckedPostFormUrlEnc']('foo', data, 'bar', tokenRef);
        expect(SetAuthParamsSpy).toHaveBeenCalled();
        expect(SetAuthParamsSpy.mock.calls[0][1]).toBe(tokenRef);
      });
    });
    //#endregion URLENCODED POST

    //#region JSON POST
    describe('#TypeCheckedPostJSON()', () => {
      const mockRes = { root: true } as IBaseMessage;

      it('should perform a HTTP post request', async () => {
        const data = { mockKey: 'mockValue' } as const;

        nock(host)
          .post(path, data)
          .matchHeader('content-type', 'application/json')
          .reply(200, mockRes);

        const response = await instance['TypeCheckedPostJson'](requestEndpoint, data, 'root', {
          accessToken: '',
        });

        expect(response.code).toBe(200);
      });

      it('should return the mocked response', async () => {
        const data = { mockKey: 'mockValue' } as const;

        nock(host)
          .post(path, data)
          .matchHeader('content-type', 'application/json')
          .reply(200, mockRes);

        const response = await instance['TypeCheckedPostJson'](requestEndpoint, data, 'root', {
          accessToken: '',
        });

        expect(response.data?.root).toBe(true);
        expect(response.code).toBe(200);
        expect(response.error).toBeUndefined();
      });

      it('should validate the error against the schema', async () => {
        const data = { mockKey: 'mockValue' } as const;

        nock(host)
          .post(path, data)
          .matchHeader('content-type', 'application/json')
          .reply(500, { code: 1, error: 'test' } as IErrorMessage);

        const response = await instance['TypeCheckedPostJson'](requestEndpoint, data, 'root', {
          accessToken: '',
        });

        expect(response.error).not.toBeUndefined();
      });

      it('should call #SetAuthParams()', () => {
        const data = { mockKey: 'mockValue' } as const;
        const SetAuthParamsSpy = jest.spyOn(instance, 'SetAuthParams' as never);
        const tokenRef = { accessToken: 'baz' } as const;

        instance['TypeCheckedPostJson']('foo', data, 'bar', tokenRef);
        expect(SetAuthParamsSpy).toHaveBeenCalled();
        expect(SetAuthParamsSpy.mock.calls[0][1]).toBe(tokenRef);
      });
    });
    //#endregion JSON POST
  });

  describe('throttling', () => {
    beforeEach(() => nock(host).get(path).reply(200));

    it('will throttle the requests', async () => {
      const requests = 5;
      const delay = 50;
      const tolerance = 0.01;

      const lowerBound = (requests - 1) * delay * (1 - tolerance);

      const instance = new ApiConsumer(
        new Validator(schema),
        new TicketVendor({ simultaneous: 1, spacing: delay })
      );

      const end = timeDifference();
      let lastPromise!: Promise<unknown>;
      for (let i = 0; i < requests; i++) {
        lastPromise = instance['TypeCheckedGet'](requestEndpoint, 'root', { accessToken: '' });
      }
      await lastPromise;

      expect(end()).toBeGreaterThan(lowerBound);
    });

    it('allows a burst of requests', async () => {
      const upperTolerance = 1.1;

      const instance = new ApiConsumer(
        new Validator(schema),
        new TicketVendor({ simultaneous: 100, spacing: 0 })
      );

      const requests = 25;
      const timeout = 50;
      const concurrency = 100;
      const expectedTime = Math.ceil(requests / concurrency) * timeout;

      const end = timeDifference();
      await Promise.all(
        Array.from({ length: requests }, () =>
          instance['TypeCheckedGet'](requestEndpoint, 'root', { accessToken: '' })
        )
      );
      expect(end()).toBeLessThan(expectedTime * upperTolerance);
    });
  });
});
