import HttpAgent, { HttpsAgent } from 'agentkeepalive';
import axios, { AxiosError, AxiosRequestConfig, AxiosResponse } from 'axios';
import * as FormData from 'form-data';
import { stringify } from 'query-string';

import { IThrottler } from '../throttling';
import { Validator } from '../validation';

export { FormData };
export type RequestConfig = AxiosRequestConfig;

export enum APIResultCode {
  Success = 'Success',
  BadData = 'Incomplete Data',
  APIError = 'API Error',
  HTTPError = 'HTTP Error',
}

export interface IRequestResponse<T, APIError> {
  result: APIResultCode;
  code: number;
  message: string; // Exception message
  data?: T; // Result data if it was successfully type checked
  error?: APIError; // Error if the error was correctly type checked
  headers?: Record<string, string>;
  body?: T | APIError; // Raw Body
}

export abstract class APIBase<BaseMessage, APIError, TokenType> {
  private readonly _AXIOS_INSTANCE = axios.create({
    httpAgent: new HttpAgent({ keepAlive: true }),
    httpsAgent: new HttpsAgent({ keepAlive: true }),
    maxRedirects: 10,
    timeout: 10e3,
  });

  public constructor(
    protected jsonValidator: Validator,
    protected throttler?: IThrottler,
    private _errorRef?: string
  ) {
    if (!(jsonValidator instanceof Validator)) {
      throw new TypeError('No JSON validator provided for api base.');
    }

    if (typeof _errorRef !== 'undefined') {
      if (typeof _errorRef !== 'string') throw new TypeError('errorRef must be a string.');
      if (_errorRef.length === 0) throw new Error('Invalid reference for errorRef.');
    }
  }

  /**
   * @description Performs a HTTP GET request with content-type: application/json and validates the response against a schema.
   * @param url The url to request.
   * @param schemaRef The reference key for the object in the schema to validate against.
   * @param token An API token object to use for authorisation in `setAuthParams`.
   */
  protected async TypeCheckedGet<T extends BaseMessage>(
    url: string,
    schemaRef: string,
    token: TokenType
  ): Promise<IRequestResponse<T, APIError>> {
    const options: RequestConfig = {
      responseType: 'json',
      url,
    };

    this.SetAuthParams(options, token);

    return this._TypeCheckedRequestInternal<T>(options, schemaRef);
  }

  /**
   * @description Performs a HTTP POST request and validates the response against a schema.
   * @param url The url to request.
   * @param formData A string to send as form data.
   * @param schemaRef The reference key for the object in the schema to validate against.
   * @param token An API token object to use for authorisation in `setAuthParams`.
   */
  protected async TypeCheckedPostForm<T extends BaseMessage>(
    url: string,
    formData: FormData,
    schemaRef: string,
    token?: TokenType
  ): Promise<IRequestResponse<T, APIError>> {
    const options: RequestConfig = {
      data: formData,
      headers: formData.getHeaders(),
      method: 'post',
      responseType: 'json',
      url,
    };

    this.SetAuthParams(options, token);

    return this._TypeCheckedRequestInternal<T>(options, schemaRef);
  }

  /**
   * @description Performs a HTTP POST request and validates the response against a schema.
   * @param url The url to request.
   * @param formData An object that gets converted into a querystring for form data.
   * @param schemaRef The reference key for the object in the schema to validate against.
   * @param token An API token object to use for authorisation in `setAuthParams`.
   */
  protected async TypeCheckedPostFormUrlEnc<T extends BaseMessage>(
    url: string,
    formData: Record<string, string>,
    schemaRef: string,
    token?: TokenType
  ): Promise<IRequestResponse<T, APIError>> {
    const options: RequestConfig = {
      data: stringify(formData),
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded',
      },
      method: 'post',
      responseType: 'json',
      url,
    };

    this.SetAuthParams(options, token);

    return this._TypeCheckedRequestInternal<T>(options, schemaRef);
  }

  /**
   * @description Performs a HTTP POST request and validates the response against a schema.
   * @param url The url to request.
   * @param data A JSON object to serialize as form data.
   * @param schemaRef The reference key for the object in the schema to validate against.
   * @param token An API token object to use for authorisation in `setAuthParams`.
   */
  protected async TypeCheckedPostJson<T extends BaseMessage, PostType>(
    url: string,
    data: PostType,
    schemaRef: string,
    token?: TokenType
  ): Promise<IRequestResponse<T, APIError>> {
    const options: RequestConfig = {
      data: JSON.stringify(data),
      headers: {
        'Content-Type': 'application/json',
      },
      method: 'post',
      responseType: 'json',
      url,
    };

    this.SetAuthParams(options, token);

    return this._TypeCheckedRequestInternal<T>(options, schemaRef);
  }

  /**
   * @description Performs a request and validates it against the schema.
   * @returns A `Promise` which resolves to an object containing the response and the status.
   */
  private async _TypeCheckedRequestInternal<T extends BaseMessage>(
    options: AxiosRequestConfig,
    schemaRef: string
  ): Promise<IRequestResponse<T, APIError>> {
    // Create return object
    const meta: IRequestResponse<T, APIError> = {
      code: 0,
      message: '',
      result: APIResultCode.HTTPError,
    };

    const release = await this.throttler?.Await();

    let response: AxiosResponse<T>;
    try {
      // Make request
      response = await this._AXIOS_INSTANCE.request(options);

      /* Set Meta Fields */
      meta.headers = response.headers as Record<string, string>;
      meta.code = response.status;
      meta.body = response.data;

      /* Validate Body */
      const { valid: isValid, errors } = this.jsonValidator.Validate(
        response.data,
        schemaRef,
        false
      );
      if (isValid) {
        meta.result = APIResultCode.Success;
        meta.data = response.data;
      } else {
        meta.result = APIResultCode.BadData;
        meta.message = errors.join(', ');
      }
    } catch (e) {
      const err = e as AxiosError<APIError>;

      if (typeof err.response !== 'undefined') {
        const { status, data } = err.response;

        /* Set Meta Fields */
        meta.headers = err.response.headers as Record<string, string>;
        meta.code = status ?? 0;
        meta.body = data;

        /* If we have a schema reference for the error, then typecheck the error
         and store it in the response */
        if (typeof this._errorRef !== 'undefined') {
          const { valid: isValid } = this.jsonValidator.Validate(data, this._errorRef, false);

          if (isValid) {
            meta.result = APIResultCode.APIError;
            meta.error = data;
          }
        }

        // We did not get a successful HTTP response
        meta.message = err.message;
      }
    } finally {
      if (release) release();
    }

    return meta;
  }

  /**
   * @description Abstract function that get's called for every request for the API implementation to set parameters based on the supplied token.
   * @param config An axios request config to modify.
   * @param token A token used to modify the `config`. This is derived from the `token` parameter for the APIBase methods.
   */

  protected abstract SetAuthParams(config: RequestConfig, token?: TokenType): void;
}
