import { FilteredObject } from './types/filtered';

/**
 * @description Plucks values associated with `keys` from `object`. Can trim string keys.
 * @param object The source object to pull key-value pairs from.
 * @param keys An array of keys from `object` to pluck.
 * @param trim Whether to trim string keys on parse, default = false.
 * @returns A new object with plucked key-value pairs from `object` which has keys from the restricted set: `keys`.
 */
export function FilterObject<O extends object, K extends (keyof O)[], T extends boolean>(
  object: O,
  keys: K,
  trim: T = false as T
): FilteredObject<O, K, T> {
  return keys.reduce<Partial<FilteredObject<O, K, T>>>((plucked, key) => {
    if (Object.prototype.hasOwnProperty.call(object, key)) {
      return { ...plucked, [trim && typeof key === 'string' ? key.trim() : key]: object[key] };
    }
    return plucked;
  }, {}) as FilteredObject<O, K, T>;
}
