import chalk from 'chalk';
import merge from 'lodash.merge';
import logform from 'logform';
import winston from 'winston';

import type { ILoggerConfig } from './types/options';

export interface ILogger extends winston.Logger {
  child(overrides?: ILoggerConfig): ILogger;
}

export function CreateLogger(cfg: ILoggerConfig = {}): ILogger {
  /* Formatters */
  const formatters: logform.Format[] = [winston.format.splat()];

  /* Timestamp */
  if (cfg?.timestamps) formatters.push(winston.format.timestamp());

  /* Colors */
  let label = cfg?.label?.name;
  if (label) {
    if (cfg?.colors) {
      const color = cfg.label!.color;
      if (color) label = chalk[color](label);
    }
    formatters.push(winston.format.label({ label }));
  }

  if (cfg?.colors) formatters.push(winston.format.colorize());

  /* Formatting */
  formatters.push(
    winston.format.printf(({ timestamp, level, message, label }: logform.TransformableInfo) => {
      let fullMessage = `${level} ${message}`;

      if (typeof timestamp === 'string') {
        fullMessage = `${timestamp.slice(0, 19).replace('T', ' ')} ${fullMessage}`;
      }

      if (typeof label !== 'undefined') fullMessage = `[${label}] ${fullMessage}`;

      return fullMessage;
    })
  );

  /* JSON */
  if (cfg?.json) formatters.push(winston.format.json());

  const { colors: _shouldColorise, ...truncatedOpts } = cfg;
  const logger = winston.createLogger({
    ...truncatedOpts,
    format: winston.format.combine(...formatters),
    transports: [new winston.transports.Console()],
  });
  logger.child = (overrides?: ILoggerConfig) => CreateLogger(merge(cfg, overrides));
  return logger;
}
