import { EventEmitter } from 'eventemitter3';

import type { ILogger } from '../logger';
import { Sleeper } from '../time/sleeper';
import type { ISchedulerConfig } from './types/config';

export interface ISchedulerEvents<R> {
  prerun: [];
  postrun: [value: R];
  error: [error: Error];
}

export class Scheduler<R> extends EventEmitter<ISchedulerEvents<R>> {
  protected sleeper?: Sleeper;
  protected isRunning = false;

  /**
   * @description Schedules a task to be run at a given interval - reporting its status through events.
   * @param cfg.task The function to call.
   * @param cfg.interval How long to wait between calls of `task`, specified in milliseconds.
   */
  public constructor(protected config: ISchedulerConfig<R>, protected logger?: ILogger) {
    super();

    if (config.interval < 100) {
      this.logger?.warn('A low poll time may result in application instability.');
    }
  }

  /**
   * @returns True if the scheduler is running.
   */
  public IsRunning(): boolean {
    return this.isRunning;
  }

  /**
   * @description Starts the scheduler.
   */
  public async Start(): Promise<void> {
    if (this.isRunning) {
      return;
    }

    this.isRunning = true;

    while (this.isRunning) {
      await this._DoTask();
      this.sleeper = new Sleeper(this.config.interval);
      await this.sleeper.Wait();
    }
  }

  /**
   * @description Executes the scheduled task immediately.
   */
  public async Execute(): Promise<void> {
    if (this.isRunning) {
      this.sleeper?.End();
    } else {
      await this._DoTask();
    }
  }

  /**
   * @description Stops the scheduler from executing tasks.
   */
  public Stop(): void {
    if (this.isRunning) {
      this.isRunning = false;
    }
  }

  private async _DoTask(): Promise<void> {
    try {
      this.emit('prerun');
      this.logger?.debug('Executing task');

      let response = this.config.task();
      if (response instanceof Promise) {
        response = await response;
      }

      this.emit('postrun', response);
    } catch (err) {
      this.logger?.error(`Error executing task: ${(err as Error).stack}`);
      this.emit('error', err);
    }
  }
}
