import type { Comparator } from './types/comparator';

export class BinaryHeap<T> {
  protected data: T[] = [];

  /**
   * @description A fast sorted data structure based on binary trees.
   */
  public constructor(public readonly comparator: Comparator<T>) {}

  /**
   * @description Pushes a new item onto the heap.
   */
  public Queue(value: T): void {
    this.data.push(value);
    this._BubbleUp(this.data.length - 1);
  }

  /**
   * @description Removes one item from the heap.
   */
  public Dequeue(): T {
    const ret = this.data[0];
    const last = this.data.pop();

    if (this.data.length > 0 && last !== undefined) {
      this.data[0] = last;
      this._BubbleDown(0);
    }

    return ret;
  }

  /**
   * @description Returns a reference to the item at the top of the heap without removing it.
   */
  public Peek(): T {
    return this.data[0];
  }

  /**
   * @description Removes all items in the heap.
   */
  public Clear(): void {
    this.data.length = 0;
  }

  /**
   * @description Reallocates items from the bottom of the heap moving upwards.
   */
  private _BubbleUp(pos: number): void {
    while (pos > 0) {
      const parent = (pos - 1) >>> 1;

      if (this.comparator(this.data[pos], this.data[parent]) < 0) {
        const x = this.data[parent];
        this.data[parent] = this.data[pos];
        this.data[pos] = x;
        pos = parent;
      } else {
        break;
      }
    }
  }

  /**
   * @description Reallocates items from the top of the heap moving downwards.
   */
  private _BubbleDown(pos: number): void {
    const last = this.data.length - 1;

    // eslint-disable-next-line no-constant-condition
    while (true) {
      const left = (pos << 1) + 1;
      const right = left + 1;

      let minIndex = pos;

      if (left <= last && this.comparator(this.data[left], this.data[minIndex]) < 0) {
        minIndex = left;
      }

      if (right <= last && this.comparator(this.data[right], this.data[minIndex]) < 0) {
        minIndex = right;
      }

      if (minIndex !== pos) {
        const x = this.data[minIndex];
        this.data[minIndex] = this.data[pos];
        this.data[pos] = x;
        pos = minIndex;
      } else {
        break;
      }
    }
  }
}
