import { BinaryHeap } from '../binary-heap';
import type { ITaskWrapper } from './types/task';

export class PriorityQueue<T> {
  protected readonly strategy = new BinaryHeap<ITaskWrapper<T>>(
    (a, b) => a.priority - b.priority || a.added - b.added
  );
  protected size = 0;

  public get length(): number {
    return this.size;
  }

  /**
   * @description An ordered queue backed by a binary heap.
   */
  public constructor() {
    //
  }

  /**
   * @description Pushes a new item into the queue.
   * @param priority Specifies the importance of `value`. Higher is more important.
   * @param value The value to store in the queue.
   */
  public Queue(priority = 5, value: T): void {
    this.size++;
    this.strategy.Queue({ added: process.hrtime.bigint(), priority, value });
  }

  /**
   * @description Returns one item from the queue.
   *
   * This will throw an error if the queue is empty.
   */
  public Dequeue(): T {
    if (!this.size) throw new Error('Queue is empty.');
    this.size--;
    return this.strategy.Dequeue().value;
  }

  /**
   * @description Returns a reference to the item at the top of the heap without removing it.
   *
   * This will throw an error if the queue is empty.
   */
  public Peek(): T {
    if (!this.size) throw new Error('Queue is empty.');
    return this.strategy.Peek().value;
  }

  /**
   * @description Removes all items in the queue.
   */
  public Clear(): void {
    this.size = 0;
    this.strategy.Clear();
  }
}
