export class RoundRobin<T> {
  protected store: Set<T>;
  protected items: readonly T[] = [];
  protected index = 0;

  /**
   * @description An extension of `Set<T>` that provides a circular `IterableIterator<T>` for value rotating.
   * @param values An `Iterable<T>` of values to construct the Set<T> with.
   */
  public constructor(values?: Iterable<T>) {
    this.store = new Set<T>(values);
    this.UpdateItems();
  }

  /**
   * @param value The value to add to the queue.
   * @returns The queue instance.
   */
  public Add(value: T): this {
    this.store.add(value);
    this.UpdateItems();
    return this;
  }

  /**
   * @param value The value to check.
   * @returns True if `value` exists in the queue.
   */
  public Has(value: T): boolean {
    return this.store.has(value);
  }

  /**
   * @param value The value to remove from the queue.
   * @returns True if the item existed in the queue and was deleted.
   */
  public Delete(value: T): boolean {
    const didDelete = this.store.delete(value);
    this.UpdateItems();
    return didDelete;
  }

  /**
   * @description Removes all items from the queue.
   */
  public Clear(): void {
    this.store.clear();
    this.UpdateItems();
  }

  /**
   * @description Retrieves the next item in the queue.
   */
  public GetNext(): T {
    if (this.index >= this.store.size) this.index = 0;
    const value = this.items[this.index];
    ++this.index;
    return value;
  }

  /**
   * @private
   * @description Updates the queue from the values currently in the `Set<T>`.
   */
  protected UpdateItems(): void {
    this.items = [...this.store.values()];
  }
}
