import { Sleeper } from '../../time/sleeper';
import { SyncList } from './sync-list';

function generateList(length: number): number[] {
  return Array.from({ length }, (k, i) => i);
}

describe('SyncList', () => {
  let addItem: jest.Mock;
  let deleteItem: jest.Mock;
  let syncList: SyncList<number, string>;

  beforeAll(function () {
    addItem = jest.fn();
    deleteItem = jest.fn();
  });

  beforeEach(() => {
    addItem.mockReset();
    deleteItem.mockReset();
    syncList?.removeAllListeners();
    syncList = new SyncList<number, string>(addItem, deleteItem);
  });

  describe('methods', () => {
    describe('#Sync()', () => {
      it('should not update if the list is unchanged', () => {
        expect(syncList.GetElements()).toEqual([]);
        const items = generateList(50);
        expect(syncList.Sync(items)).toBeTruthy();
        expect(syncList.Sync([...items])).toBeFalsy();
      });

      it('should sync if the list has been updated', () => {
        expect(syncList.GetElements()).toEqual([]);
        const items = generateList(50);
        expect(syncList.Sync(items)).toBeTruthy();
        expect(syncList.Sync(items.reverse())).toBeTruthy();
      });
    });

    describe('#GetElements()', () => {
      it('should return no elements when newly initialised', () => {
        expect(syncList.GetElements()).toEqual([]);
        /* check 'holey' arrays like [,,,,,,,] */
        expect(syncList.GetElements()).toHaveLength(0);
      });

      it('should return elements after insert', () => {
        expect(syncList.Sync(generateList(50))).toBeTruthy();
        expect(syncList.GetElements()).not.toEqual([]);
        expect(syncList.GetElements().length).toBeGreaterThan(0);
      });
    });

    describe('#Clear()', () => {
      it('should clear the items in the list', () => {
        expect(syncList.Sync(generateList(50))).toBeTruthy();
        expect(syncList.GetElements().length).toBeGreaterThan(0);
        syncList.Clear();
        expect(syncList.GetElements()).toHaveLength(0);
      });
    });
  });

  describe('events', () => {
    describe('@elementAdded', () => {
      it('should not be emitted for a false sync', async () => {
        expect(syncList.GetElements()).toHaveLength(0);

        let didRun = false;
        let earlyEnd!: () => void;
        const taskPromise = new Promise<void>(r => (earlyEnd = r));

        syncList.on('elementAdded', () => {
          didRun = true;
          earlyEnd();
        });
        syncList.Sync([]);
        expect(syncList.Sync([])).toBeFalsy();
        await Promise.race([Sleeper.Sleep(100), taskPromise]);

        expect(didRun).toBeFalsy();
      });

      it('should be emitted when an item is changed', async () => {
        expect(syncList.GetElements()).toHaveLength(0);

        let didRun = false;
        let earlyEnd!: () => void;
        const taskPromise = new Promise<void>(r => (earlyEnd = r));

        syncList.on('elementAdded', () => {
          didRun = true;
          earlyEnd();
        });
        expect(syncList.Sync(generateList(5))).toBeTruthy();
        await Promise.race([Sleeper.Sleep(100), taskPromise]);

        expect(didRun).toBeTruthy();
      });

      it('should be emitted for every item change', async () => {
        const itemsToAdd = 50;

        expect(syncList.GetElements()).toHaveLength(0);

        let earlyEnd!: () => void;
        const taskPromise = new Promise<void>(r => (earlyEnd = r));
        const cb = jest.fn(earlyEnd);

        syncList.on('elementAdded', cb);
        expect(syncList.Sync(generateList(itemsToAdd))).toBeTruthy();
        await Promise.race([Sleeper.Sleep(100), taskPromise]);

        expect(cb.mock.calls).toHaveLength(itemsToAdd);
      });
    });

    describe('@elementRemoved', () => {
      it('should not be emitted for a false sync', async () => {
        expect(syncList.GetElements()).toHaveLength(0);

        let didRun = false;
        let earlyEnd!: () => void;
        const taskPromise = new Promise<void>(r => (earlyEnd = r));

        syncList.on('elementRemoved', () => {
          didRun = true;
          earlyEnd();
        });
        syncList.Sync([]);
        expect(syncList.Sync([])).toBeFalsy();
        await Promise.race([Sleeper.Sleep(100), taskPromise]);

        expect(didRun).toBeFalsy();
      });

      it('should be emitted when an item is changed', async () => {
        expect(syncList.GetElements()).toHaveLength(0);
        expect(syncList.Sync(generateList(5))).toBeTruthy();

        let didRun = false;
        let earlyEnd!: () => void;
        const taskPromise = new Promise<void>(r => (earlyEnd = r));

        syncList.on('elementRemoved', () => {
          didRun = true;
          earlyEnd();
        });
        expect(syncList.Sync([])).toBeTruthy();
        await Promise.race([Sleeper.Sleep(100), taskPromise]);

        expect(didRun).toBeTruthy();
      });

      it('should be emitted for every item change', async () => {
        const itemsToAdd = 50;

        expect(syncList.GetElements()).toHaveLength(0);
        expect(syncList.Sync(generateList(itemsToAdd))).toBeTruthy();

        let earlyEnd!: () => void;
        const taskPromise = new Promise<void>(r => (earlyEnd = r));
        const cb = jest.fn(earlyEnd);

        syncList.on('elementRemoved', cb);
        expect(syncList.Sync([])).toBeTruthy();
        await Promise.race([Sleeper.Sleep(100), taskPromise]);

        expect(cb.mock.calls).toHaveLength(itemsToAdd);
      });
    });

    describe('@listUpdated', () => {
      it('should not be emitted for a false sync', async () => {
        expect(syncList.Sync([1])).toBeTruthy();

        let earlyEnd!: () => void;
        const taskPromise = new Promise<void>(r => (earlyEnd = r));
        const cb = jest.fn(earlyEnd);

        syncList.on('listUpdated', cb);
        expect(syncList.Sync([1])).toBeFalsy();
        await Promise.race([Sleeper.Sleep(100), taskPromise]);

        expect(cb).not.toHaveBeenCalled();
      });

      it('should be emitted when an item is added', async () => {
        expect(syncList.Sync([1])).toBeTruthy();

        let earlyEnd!: () => void;
        const taskPromise = new Promise<void>(r => (earlyEnd = r));
        const cb = jest.fn(earlyEnd);

        syncList.on('listUpdated', cb);
        expect(syncList.Sync([1, 2])).toBeTruthy();
        await Promise.race([Sleeper.Sleep(100), taskPromise]);

        expect(cb).toHaveBeenCalled();
      });

      it('should be emitted when an item is removed', async () => {
        expect(syncList.Sync([1])).toBeTruthy();

        let earlyEnd!: () => void;
        const taskPromise = new Promise<void>(r => (earlyEnd = r));
        const cb = jest.fn(earlyEnd);

        syncList.on('listUpdated', cb);
        expect(syncList.Sync([])).toBeTruthy();
        await Promise.race([Sleeper.Sleep(100), taskPromise]);

        expect(cb).toHaveBeenCalled();
      });

      it('should be emitted when an item is changed', async () => {
        expect(syncList.Sync([1])).toBeTruthy();

        let earlyEnd!: () => void;
        const taskPromise = new Promise<void>(r => (earlyEnd = r));
        const cb = jest.fn(earlyEnd);

        syncList.on('listUpdated', cb);
        expect(syncList.Sync([2])).toBeTruthy();
        await Promise.race([Sleeper.Sleep(100), taskPromise]);

        expect(cb).toHaveBeenCalled();
      });
    });
  });
});
