import { EventEmitter } from 'eventemitter3';
import isEqual from 'lodash.isequal';

export type CreateFn<DataType, ElemType> = (data: DataType) => ElemType;
export type RemoveFn<ElemType> = (elem: ElemType) => void;

export interface ISyncListEvents<ElemType> {
  elementAdded: [elem: ElemType, index: number];
  elementRemoved: [elem: ElemType, index: number];
  listUpdated: [];
}

export class SyncList<DataType, ElemType> extends EventEmitter<ISyncListEvents<ElemType>> {
  protected readonly list: [DataType, ElemType][] = [];

  public constructor(
    public readonly addCb: CreateFn<DataType, ElemType>,
    public readonly removeCB: RemoveFn<ElemType>
  ) {
    super();
  }

  /**
   * @description Syncs a new list against the existing cached list.
   *
   *              Propagates updates through events and callbacks for those that were updated.
   *
   *              On Add:
   *              - Calls addCb(DataType)
   *              - Emits 'elementAdded' with payload (DataType, index)
   *
   *              On Remove:
   *              - Calls removeCb(ElemType)
   *              - Emits 'elementRemoved' with payload (DataType, index)
   *
   *              Upon Any Update:
   *              - Emits 'listUpdated' once.
   * @param newList The new list to sync against.
   */
  public Sync(newList: DataType[]): boolean {
    const len = Math.max(this.list.length, newList.length);
    let didUpdate = false;

    for (let i = 0; i < len; ++i) {
      const a: DataType | undefined = newList[i];
      const b: [DataType, ElemType] | undefined = this.list[i];

      if (!isEqual(a, b?.[0])) {
        // Needs update
        didUpdate = true;

        if (b !== undefined) {
          this.removeCB(b[1]);
          this.emit('elementRemoved', b[1], i);
        }

        if (a !== undefined) {
          const newElem = this.addCb(a);
          this.list[i] = [a, newElem];
          this.emit('elementAdded', newElem, i);
        }
      }
    }

    this.list.length = newList.length;

    if (didUpdate) this.emit('listUpdated');

    return didUpdate;
  }

  /**
   * @description Maps all DataTypes from the sync list cache into it's respective ElemType.
   * @returns Returns a list of all elements in the cached sync list.
   */
  public GetElements(): ElemType[] {
    return this.list.map(l => l[1]);
  }

  /**
   * @description Deletes all items in the cached sync list.
   * @returns True if items were cleared.
   */
  public Clear(): boolean {
    return this.Sync([]);
  }
}
