import { HelixThrottler } from './helix-throttler';

describe('HelixThrottler', () => {
  it('should await the reset time when there are no requests left', () => {
    const throttler = new HelixThrottler();
    throttler.SetRemaining('0');
    throttler.SetReset((Math.floor(Date.now() / 1000) + 3).toString());
    expect(throttler.Calculate()).toBe(3000);
  });

  it('should return 0 when there are no requests used', () => {
    const throttler = new HelixThrottler();
    throttler.SetRemaining('800');
    throttler.SetLimit('800');
    expect(throttler.Calculate()).toBe(0);
  });

  it('should return an evenly spread out delay to meet the reset count', () => {
    const throttler = new HelixThrottler();
    throttler.SetLimit('50');

    throttler.SetRemaining('2');
    throttler.SetReset((Math.floor(Date.now() / 1000) + 1).toString());
    expect(throttler.Calculate()).toBe(500);

    throttler.SetRemaining('10');
    throttler.SetReset((Math.floor(Date.now() / 1000) + 2).toString());
    expect(throttler.Calculate()).toBe(200);
  });

  it('should await the correct amount of time', async () => {
    const throttler = new HelixThrottler();
    throttler.SetLimit('50');

    const now = Date.now();
    throttler.SetRemaining('100');
    throttler.SetReset((Math.floor(now / 1000) + 1).toString());
    await throttler.Await();

    // There's bound to be a few milliseconds of inaccuracy with node, just remove it.
    const d = Math.floor((Date.now() - now) / 10) * 10;
    expect(d).toBeGreaterThanOrEqual(10);
    expect(d).toBeLessThanOrEqual(30);
  });

  it('should respectfully increase the delay as we use more requests', () => {
    const throttler = new HelixThrottler();
    throttler.SetRemaining('50');
    throttler.SetReset((Math.floor(Date.now() / 1000) + 1).toString());

    expect(throttler.Calculate()).toBe(20);

    for (let i = 0; i < 25; ++i) {
      // use 25 requests (half of our remaining)
      throttler.Invoke();
    }

    expect(throttler.Calculate()).toBe(40);
  });
});
