import clamp from 'lodash.clamp';

import type { ILogger } from '../../logger';
import { Sleeper } from '../../time/sleeper';
import type { IThrottler } from '../types/throttler';

/**
 * @description Throttles helix requests based on the header ratelimits.
 */
export class HelixThrottler implements IThrottler {
  protected remaining = 0;
  protected reset = 0;
  protected limit = 800;

  public constructor(protected readonly logger?: ILogger) {}

  public get Remaining(): number {
    return this.remaining;
  }

  public get Limit(): number {
    return this.limit;
  }

  /**
   * @description Subtracts one from the count of requests left in the time period.
   */
  public Invoke(): void {
    --this.remaining;
  }

  /**
   * @returns A `Promise` that resolves when the next task can be issued.
   */
  public async Await(): Promise<void> {
    const sleepDuration = this.Calculate();
    if (sleepDuration > 0) await Sleeper.Sleep(sleepDuration);
    this.Invoke();
  }

  /**
   * @returns The amount of time until the next request can be sent.
   */
  public Calculate(): number {
    // Make it up to a second resolution.
    const now = Math.floor(Date.now() / 1000) * 1000;
    if (this.reset < now) {
      // In the case we haven't queried the API in a while, fix our limit.
      // Take old reset, add 60 seconds * number of buckets we missed.
      this.reset = this.reset + 60 * 1000 * Math.ceil(((now - this.reset) / 60) * 1000);
      this.remaining = this.Limit;
    }

    if (this.remaining === 0) {
      // We're out of requests, await till we have some more.
      return this.reset - now;
    }

    return this.remaining === this.Limit
      ? 0
      : clamp((this.reset - now) / this.remaining, 0, 60 * 1000);
  }

  public SetReset(reset: string): void {
    const resetInt = parseInt(reset, 10);
    if (isNaN(resetInt)) return;
    if (this.reset === resetInt) return;
    this.reset = resetInt * 1000;
    this.logger?.info(`Set Reset = ${resetInt}`);
  }

  public SetRemaining(remaining: string): void {
    const remainingInt = parseInt(remaining, 10);
    if (isNaN(remainingInt)) {
      throw new RangeError("Argument for parameter 'remaining' must be an integer.");
    }
    if (this.remaining === remainingInt) return;
    this.remaining = remainingInt;
    this.logger?.info(`Set Remaining = ${remainingInt}`);
  }

  public SetLimit(limit: string): void {
    const limitInt = parseInt(limit, 10);
    if (isNaN(limitInt)) throw new RangeError("Argument for parameter 'limit' must be an integer.");
    if (this.limit === limitInt) return;
    this.limit = limitInt;
    this.logger?.info(`Set Limit = ${limit}`);
  }
}
