import { timeDifference } from '../../time/difference';
import { Sleeper } from '../../time/sleeper';
import { TicketVendor } from './ticket-vendor';

describe('TicketVendor', () => {
  describe('methods', () => {
    describe('#Await()', () => {
      it('should ratelimit queued requests', async () => {
        const instance = new TicketVendor({ simultaneous: 1, spacing: 0 });
        const awaiter = instance.Await();

        /* task 1 */
        expect(awaiter).toBeInstanceOf(Promise);
        const release = await awaiter;
        expect(typeof release).toBe('function');

        /* 2nd concurrent task */
        const m = jest.fn();
        instance.Await().then(m);
        expect(m).not.toHaveBeenCalled();
        release();
        await Sleeper.Sleep(50);
        expect(m).toHaveBeenCalled();
      });

      it('should allow bursts', async () => {
        const COUNT = 100;
        const instance = new TicketVendor({ simultaneous: COUNT, spacing: 0 });

        const end = timeDifference();
        await Promise.all(Array.from({ length: COUNT }, () => instance.Await()));
        expect(end()).toBeLessThan(COUNT);
      });

      it('should allow concurrency', async () => {
        const COUNT = 25;
        const TIMEOUT = 20;
        const CONCURRENCY = 5;

        const instance = new TicketVendor({ simultaneous: CONCURRENCY, spacing: TIMEOUT });
        const expectedTime = Math.ceil(COUNT / CONCURRENCY) * TIMEOUT;

        const end = timeDifference();
        await Promise.all(
          Array.from({ length: COUNT }, () =>
            instance.Await().then(r => Sleeper.Sleep(TIMEOUT).then(r))
          )
        );
        expect(end()).toBeGreaterThan(expectedTime);
      });
    });
  });

  describe('mutex', () => {
    const instance = new TicketVendor({ simultaneous: 1, spacing: 0 });

    it('should allow only a single task to execute', async () => {
      const TIMEOUT = 50;
      const RUNS = 5;

      const end = timeDifference();

      let lastPromise!: Promise<void>;
      for (let i = 0; i < RUNS; i++) {
        lastPromise = instance.Await().then(k => Sleeper.Sleep(TIMEOUT).then(k));
      }
      await lastPromise;

      expect(end()).toBeGreaterThan(RUNS * TIMEOUT);
    });
  });
});
