import { Semaphore } from 'await-semaphore';

import type { ILogger } from '../../logger';
import { Sleeper } from '../../time/sleeper';
import type { IThrottler } from '../types/throttler';
import type { ITicketVendorConfig } from './types/config';

export class TicketVendor implements IThrottler {
  protected readonly semaphore: Semaphore;
  protected lastIssued = 0;
  protected queued = 0;
  protected active = 0;

  /**
   * @description A factory that returns an instance of `TicketVendor`.
   * @param simultaneous How many tickets to handle concurrently.
   * @param spacing How long to wait between issuing tickets, specified in milliseconds.
   */
  public constructor(public Config: ITicketVendorConfig, protected readonly Logger?: ILogger) {
    /* simultaneous */
    if (typeof Config.simultaneous !== 'number') {
      throw new TypeError('No value for simultaneous specified.');
    }

    if (isNaN(Config.simultaneous) || !Number.isSafeInteger(Config.simultaneous)) {
      throw new RangeError('Invalid value for simultaneous specified.');
    }

    /* spacing */
    if (typeof Config.spacing !== 'number') throw new TypeError('No value for spacing specified.');
    if (isNaN(Config.spacing) || !Number.isSafeInteger(Config.spacing)) {
      throw new RangeError('Invalid value for spacing specified.');
    }

    if (Config.spacing < 0) throw new RangeError('spacing cannot be less than zero.');

    this.semaphore = new Semaphore(Config.simultaneous);
  }

  /**
   * @description A promise genrator that waits until a ticket is free to consume.
   * @returns A Promise which resolves to a release function you should call at the end of your ticket usage.
   */
  public async Await(): Promise<() => void> {
    this.queued++;
    const relFn = await this.semaphore.acquire();

    const now = Date.now();
    let waitTime = 0;

    const earliest = this.lastIssued + this.Config.spacing;
    if (earliest > now) {
      this.lastIssued = earliest;
      waitTime = earliest - now;
      await Sleeper.Sleep(waitTime);
    } else {
      this.lastIssued = now;
    }

    const qu = --this.queued;
    const act = ++this.active;

    this.Logger?.verbose(`Activated Ticket: ${act}/${qu}`, {
      active: act,
      delay: waitTime,
      queued: qu,
    });

    return () => {
      --this.active;
      relFn();
    };
  }
}
