export class Sleeper {
  protected readonly promise: Promise<void>;

  /**
   * @param ms How long to wait in milliseconds.
   */
  public constructor(ms: number) {
    if (typeof ms !== 'number') throw new TypeError('Invalid value for ms specified.');
    if (isNaN(ms) || !Number.isFinite(ms)) throw new RangeError('Invalid value for ms specified.');

    ms = Math.round(ms);
    if (ms <= 0) throw new RangeError('Invalid value for ms specified.');

    this.promise = new Promise<void>(resolve => {
      let timeout: NodeJS.Timeout | undefined = setTimeout(() => {
        timeout = undefined;
        resolve();
      }, ms);

      this._Kill = () => {
        if (typeof timeout !== 'undefined') {
          clearTimeout(timeout);
          resolve();
        }
      };
    });
  }

  /**
   * @description Sleeps for a specified time.
   * @param timeout How long to wait in milliseconds.
   * @returns A `Promise` that resolves after `timeout` milliseconds have elapsed.
   */
  public static Sleep(timeout: number): Promise<void> {
    return new Promise<void>(r => setTimeout(r, timeout));
  }

  /**
   * @description Resolves the timeout promise early.
   */
  public End(): void {
    this._Kill?.();
  }

  /**
   * @returns The internal promise that resolves when the time is complete.
   */
  public Wait(): Promise<void> {
    return this.promise;
  }

  /**
   * @description Set in the constructor, has a reference to the promise resolver.
   */
  private _Kill?(): void;
}
