import JSONSchema, { Schema, ValidatorResult } from 'jsonschema';

export { Schema, ValidatorResult };

export class Validator {
  protected readonly SCHEMA_VALIDATOR = new JSONSchema.Validator();

  /**
   * @param schema The JSON schema to parse and validate against.
   * @returns An instance of `JSONValidator` which validates against `schema`.
   */
  public constructor(schema: Schema) {
    this.SCHEMA_VALIDATOR.addSchema(schema);
  }

  /**
   * @description Validates an object against a typed JSON schema.
   * @param data The source object to validate.
   * @param ref The object reference key in the schema to validate against.
   * @param doThrow Whether an error should be thrown if validation fails, default=true.
   * @returns An object specifying which schema the object was validated against, and any errors or warnings during the validation.
   */
  public Validate<DataType>(data: DataType, ref: string, doThrow = true): ValidatorResult {
    const res = this.SCHEMA_VALIDATOR.validate(data, { $ref: `#/definitions/${ref}` });
    if (!res.valid && doThrow) throw new Error(res.errors.join(', '));
    return res;
  }
}
